<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;

class CryptoService
{
    private $apis = [
        'binance' => [
            'url' => 'https://api.binance.com/api/v3/ticker/24hr'
        ],
        'kucoin' => [
            'url' => 'https://api.kucoin.com/api/v1/market/allTickers'
        ],
        'huobi' => [
            'url' => 'https://api.huobi.pro/market/tickers'
        ],
        'gateio' => [
            'url' => 'https://api.gateio.ws/api/v4/spot/tickers'
        ]
    ];

    private $icons = [
        'bitcoin' => 'https://www.cryptocompare.com/media/37746251/btc.png',
        'tether' => 'https://www.cryptocompare.com/media/37746338/usdt.png',
        'ethereum' => 'https://www.cryptocompare.com/media/37746238/eth.png',
        'tron' => 'https://www.cryptocompare.com/media/37746879/trx.png',
        'polkadot' => 'https://www.cryptocompare.com/media/39334571/dot.png',
        'bitcoin-cash' => 'https://www.cryptocompare.com/media/37746245/bch.png',
        'litecoin' => 'https://www.cryptocompare.com/media/37746243/ltc.png',
        'stellar' => 'https://www.cryptocompare.com/media/37746346/xlm.png',
        'dash' => 'https://www.cryptocompare.com/media/37746893/dash.png',
        'solana' => 'https://www.cryptocompare.com/media/37747734/sol.png',
        'binancecoin' => 'https://www.cryptocompare.com/media/40485170/bnb.png'
    ];

    private $symbolMap = [
        'BTC' => 'bitcoin',
        'ETH' => 'ethereum',
        'USDT' => 'tether',
        'TRX' => 'tron',
        'BNB' => 'binancecoin',
        'DOT' => 'polkadot',
        'BCH' => 'bitcoin-cash',
        'LTC' => 'litecoin',
        'XLM' => 'stellar',
        'DASH' => 'dash',
        'SOL' => 'solana'
    ];

    private $binanceSymbols = [
        'BTCUSDT' => 'bitcoin',
        'ETHUSDT' => 'ethereum',
        'TRXUSDT' => 'tron',
        'BNBUSDT' => 'binancecoin',
        'DOTUSDT' => 'polkadot',
        'BCHUSDT' => 'bitcoin-cash',
        'LTCUSDT' => 'litecoin',
        'XLMUSDT' => 'stellar',
        'DASHUSDT' => 'dash',
        'SOLUSDT' => 'solana'
    ];

    public function getPrices()
    {
        return Cache::remember('crypto_prices', 60, function () {
            try {
                $prices = $this->fetchFromBinance();
                if (!empty($prices)) {
                    return array_merge($this->getDefaultPrices(), $prices);
                }
            } catch (\Exception $e) {
                Log::error("Error fetching prices from Binance", [
                    'error' => $e->getMessage(),
                    'trace' => $e->getTraceAsString()
                ]);
            }

            // If Binance fails, try other providers
            foreach ($this->apis as $provider => $api) {
                if ($provider === 'binance') continue; // Skip Binance as we already tried it

                try {
                    $prices = $this->fetchFromProvider($provider);
                    if (!empty($prices)) {
                        return array_merge($this->getDefaultPrices(), $prices);
                    }
                } catch (\Exception $e) {
                    Log::error("Error fetching prices from {$provider}", [
                        'error' => $e->getMessage(),
                        'trace' => $e->getTraceAsString(),
                        'api' => $api['url']
                    ]);
                    continue;
                }
            }

            Log::warning('All API providers failed, returning default prices');
            return $this->getDefaultPrices();
        });
    }

    private function fetchFromBinance()
    {
        $response = Http::timeout(5)
            ->retry(2, 100)
            ->get($this->apis['binance']['url']);
        
        if (!$response->successful()) {
            Log::warning("Failed to fetch from Binance", [
                'status' => $response->status(),
                'body' => $response->body()
            ]);
            return [];
        }

        $data = $response->json();
        if (empty($data) || !is_array($data)) {
            Log::warning("Empty or invalid response from Binance");
            return [];
        }

        $prices = [];
        foreach ($data as $ticker) {
            if (isset($this->binanceSymbols[$ticker['symbol']])) {
                $coinId = $this->binanceSymbols[$ticker['symbol']];
                $prices[$coinId] = [
                    'usd' => $this->validatePrice($ticker['lastPrice']),
                    'usd_24h_change' => $this->validatePriceChange($ticker['priceChangePercent']),
                    'last_updated' => now()->timestamp
                ];
            }
        }

        // Add USDT with fixed price
        $prices['tether'] = [
            'usd' => 1,
            'usd_24h_change' => 0,
            'last_updated' => now()->timestamp
        ];

        return $prices;
    }

    private function fetchFromProvider($provider)
    {
        $api = $this->apis[$provider];
        $response = Http::timeout(5)
            ->retry(2, 100)
            ->get($api['url']);
        
        if (!$response->successful()) {
            Log::warning("Failed to fetch from {$provider}", [
                'status' => $response->status(),
                'body' => $response->body()
            ]);
            return [];
        }

        $data = $response->json();
        if (empty($data)) {
            Log::warning("Empty response from {$provider}");
            return [];
        }

        return match($provider) {
            'kucoin' => $this->formatKucoinResponse($data),
            'huobi' => $this->formatHuobiResponse($data),
            'gateio' => $this->formatGateioResponse($data),
            default => []
        };
    }

    private function formatKucoinResponse($data)
    {
        if (!isset($data['data']['ticker']) || !is_array($data['data']['ticker'])) {
            return [];
        }

        return collect($data['data']['ticker'])
            ->filter(function ($item) {
                return str_ends_with($item['symbol'], '-USDT');
            })
            ->mapWithKeys(function ($item) {
                $symbol = explode('-', $item['symbol'])[0];
                $changeRate = floatval($item['changeRate']) * 100;
                
                $standardSymbol = $this->getStandardSymbol($symbol);
                if (!$standardSymbol) return [];

                return [
                    $standardSymbol => [
                        'usd' => $this->validatePrice($item['last']),
                        'usd_24h_change' => $this->validatePriceChange($changeRate),
                        'last_updated' => now()->timestamp
                    ]
                ];
            })
            ->toArray();
    }

    private function formatHuobiResponse($data)
    {
        if (!isset($data['data']) || !is_array($data['data'])) {
            return [];
        }

        return collect($data['data'])
            ->filter(function ($item) {
                return str_ends_with($item['symbol'], 'usdt');
            })
            ->mapWithKeys(function ($item) {
                $symbol = str_replace('usdt', '', $item['symbol']);
                $price = floatval($item['close']);
                $open = floatval($item['open']);
                $priceChange = $open > 0 ? (($price - $open) / $open) * 100 : 0;
                
                $standardSymbol = $this->getStandardSymbol($symbol);
                if (!$standardSymbol) return [];

                return [
                    $standardSymbol => [
                        'usd' => $this->validatePrice($price),
                        'usd_24h_change' => $this->validatePriceChange($priceChange),
                        'last_updated' => now()->timestamp
                    ]
                ];
            })
            ->toArray();
    }

    private function formatGateioResponse($data)
    {
        if (!is_array($data)) {
            return [];
        }

        return collect($data)
            ->filter(function ($item) {
                return str_ends_with($item['currency_pair'], '_USDT');
            })
            ->mapWithKeys(function ($item) {
                $symbol = explode('_', $item['currency_pair'])[0];
                
                $standardSymbol = $this->getStandardSymbol($symbol);
                if (!$standardSymbol) return [];

                return [
                    $standardSymbol => [
                        'usd' => $this->validatePrice($item['last']),
                        'usd_24h_change' => $this->validatePriceChange($item['change_percentage']),
                        'last_updated' => now()->timestamp
                    ]
                ];
            })
            ->toArray();
    }

    private function validatePrice($price)
    {
        $price = floatval($price);
        return $price > 0 ? $price : 0;
    }

    private function validatePriceChange($change)
    {
        $change = floatval($change);
        return max(-100, min(1000, $change));
    }

    private function getDefaultPrices()
    {
        $timestamp = now()->timestamp;
        return [
            'bitcoin' => ['usd' => 0, 'usd_24h_change' => 0, 'last_updated' => $timestamp],
            'ethereum' => ['usd' => 0, 'usd_24h_change' => 0, 'last_updated' => $timestamp],
            'tether' => ['usd' => 1, 'usd_24h_change' => 0, 'last_updated' => $timestamp],
            'tron' => ['usd' => 0, 'usd_24h_change' => 0, 'last_updated' => $timestamp],
            'binancecoin' => ['usd' => 0, 'usd_24h_change' => 0, 'last_updated' => $timestamp],
            'polkadot' => ['usd' => 0, 'usd_24h_change' => 0, 'last_updated' => $timestamp],
            'bitcoin-cash' => ['usd' => 0, 'usd_24h_change' => 0, 'last_updated' => $timestamp],
            'litecoin' => ['usd' => 0, 'usd_24h_change' => 0, 'last_updated' => $timestamp],
            'stellar' => ['usd' => 0, 'usd_24h_change' => 0, 'last_updated' => $timestamp],
            'dash' => ['usd' => 0, 'usd_24h_change' => 0, 'last_updated' => $timestamp],
            'solana' => ['usd' => 0, 'usd_24h_change' => 0, 'last_updated' => $timestamp]
        ];
    }

    public function getStandardSymbol($symbol)
    {
        return $this->symbolMap[strtoupper($symbol)] ?? null;
    }

    public function getIconUrl($symbol)
    {
        return $this->icons[strtolower($symbol)] ?? null;
    }

    public function getNetworkIcon($network)
    {
        $networkIcons = [
            'trc20' => $this->getIconUrl('tron'),
            'bep20' => $this->getIconUrl('binancecoin'),
            'erc20' => $this->getIconUrl('ethereum'),
        ];

        return $networkIcons[strtolower($network)] ?? null;
    }
}